package de.melanx.utilitix.content.shulkerboat;

import de.melanx.utilitix.config.FeatureConfig;
import net.minecraft.core.component.DataComponents;
import net.minecraft.stats.Stats;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResultHolder;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntitySelector;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.entity.vehicle.Boat;
import net.minecraft.world.flag.FeatureFlagSet;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.component.ItemContainerContents;
import net.minecraft.world.level.ClipContext;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.gameevent.GameEvent;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.HitResult;
import net.minecraft.world.phys.Vec3;
import org.moddingx.libx.base.ItemBase;
import org.moddingx.libx.mod.ModX;

import javax.annotation.Nonnull;
import java.util.List;

public class ShulkerBoatItem extends ItemBase {

    private final Boat.Type boatType;

    public ShulkerBoatItem(ModX mod, Boat.Type boatType, Properties properties) {
        super(mod, properties);
        this.boatType = boatType;
    }

    @Nonnull
    @Override
    public InteractionResultHolder<ItemStack> use(@Nonnull Level level, @Nonnull Player player, @Nonnull InteractionHand hand) {
        ItemStack stack = player.getItemInHand(hand);
        BlockHitResult hitResult = Item.getPlayerPOVHitResult(level, player, ClipContext.Fluid.ANY);
        if (hitResult.getType() == HitResult.Type.MISS) {
            return InteractionResultHolder.pass(stack);
        }

        Vec3 view = player.getViewVector(1);
        List<Entity> entities = level.getEntities(player, player.getBoundingBox().expandTowards(view.scale(5)).inflate(1), EntitySelector.NO_SPECTATORS.and(Entity::isPickable));
        if (!entities.isEmpty()) {
            Vec3 eyePos = player.getEyePosition();

            for (Entity entity : entities) {
                AABB box = entity.getBoundingBox().inflate(entity.getPickRadius());
                if (box.contains(eyePos)) {
                    return InteractionResultHolder.pass(stack);
                }
            }
        }

        if (hitResult.getType() != HitResult.Type.BLOCK) {
            return InteractionResultHolder.pass(stack);
        }

        ShulkerBoat boat = new ShulkerBoat(level, hitResult.getLocation());
        boat.setVariant(this.boatType);
        boat.setYRot(player.getYRot());
        if (!level.noCollision(boat, boat.getBoundingBox())) {
            return InteractionResultHolder.fail(stack);
        }

        if (!level.isClientSide) {
            if (stack.has(DataComponents.CONTAINER)) {
                ItemContainerContents itemContainerContents = stack.getOrDefault(DataComponents.CONTAINER, ItemContainerContents.EMPTY);
                itemContainerContents.copyInto(boat.getItemStacks());
            }

            if (stack.has(DataComponents.CUSTOM_NAME)) {
                boat.setCustomName(stack.getHoverName());
            }

            level.addFreshEntity(boat);
            level.gameEvent(player, GameEvent.ENTITY_PLACE, hitResult.getLocation());
            if (!player.getAbilities().instabuild) {
                stack.shrink(1);
            }
        }

        player.awardStat(Stats.ITEM_USED.get(this));
        return InteractionResultHolder.sidedSuccess(stack, level.isClientSide);
    }

    @Override
    public boolean canFitInsideContainerItems(@Nonnull ItemStack stack) {
        return false;
    }

    @Override
    public boolean isEnabled(@Nonnull FeatureFlagSet enabledFeatures) {
        return FeatureConfig.Transportation.shulkerBoats;
    }
}
