package de.melanx.skyblockbuilder.data;

import de.melanx.skyblockbuilder.commands.invitation.InviteCommand;
import de.melanx.skyblockbuilder.compat.minemention.MineMentionCompat;
import de.melanx.skyblockbuilder.config.common.TemplatesConfig;
import de.melanx.skyblockbuilder.util.SkyComponents;
import de.melanx.skyblockbuilder.util.WorldUtil;
import de.melanx.skyblockbuilder.world.IslandPos;
import net.minecraft.ChatFormatting;
import net.minecraft.core.BlockPos;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.nbt.Tag;
import net.minecraft.network.chat.ClickEvent;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.network.chat.Style;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.server.players.PlayerList;
import net.minecraft.world.entity.player.Player;
import net.neoforged.fml.ModList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;

public class Team {

    private static final String TEAM_ID = "team_id";
    private static final String ISLAND = "island";
    private static final String NAME = "name";
    private static final String VISITS = "visits";
    private static final String ALLOW_JOIN_REQUESTS = "allow_join_requests";
    private static final String CREATED_AT = "created_at";
    private static final String LAST_CHANGED = "last_changed";
    private static final String PLAYERS = "players";
    private static final String PLAYER = "player";
    private static final String SPAWNS = "spawns";
    private static final String DEFAULT_SPAWNS = "default_spawns";
    private static final String DIRECTION = "direction";
    private static final String JOIN_REQUESTS = "join_requests";
    private static final String ID = "player_id";
    private static final String PLACED_SPREADS = "placed_spreads";
    private static final String POS = "pos";
    private static final String SIZE = "size";

    private final SkyblockSavedData data;
    private final Set<UUID> players = new CopyOnWriteArraySet<>();
    private final Set<UUID> joinRequests = new CopyOnWriteArraySet<>();
    private final Set<TemplatesConfig.Spawn> possibleSpawns = new CopyOnWriteArraySet<>();
    private final Set<TemplatesConfig.Spawn> defaultPossibleSpawns = new CopyOnWriteArraySet<>();
    private final Map<String, Set<PlacedSpread>> placedSpreads = new ConcurrentHashMap<>();

    private UUID teamId;
    private IslandPos island;
    private String name;
    private boolean allowVisits;
    private boolean allowJoinRequests;
    private long createdAt;
    private long lastChanged;

    private Team(SkyblockSavedData data) {
        this(data, null, null);
    }

    public Team(SkyblockSavedData data, IslandPos island) {
        this(data, island, UUID.randomUUID());
    }

    public Team(SkyblockSavedData data, IslandPos island, UUID teamId) {
        this.data = data;
        this.island = island;
        this.teamId = teamId;
        this.allowVisits = false;
        this.createdAt = System.currentTimeMillis();
        this.lastChanged = System.currentTimeMillis();
    }

    public static Team create(SkyblockSavedData data, CompoundTag tag) {
        Team team = new Team(data);
        team.deserializeNBT(tag);

        return team;
    }

    public boolean isSpawn() {
        return Objects.equals(this.teamId, SkyblockSavedData.SPAWN_ID);
    }

    public String getName() {
        return this.name;
    }

    public UUID getId() {
        return this.teamId;
    }

    public void setName(String name) {
        this.name = name;
        this.updateLastChanged();
    }

    public IslandPos getIsland() {
        return this.island;
    }

    public void setIsland(IslandPos island) {
        this.island = island;
        this.updateLastChanged();
    }

    public Set<UUID> getPlayers() {
        return this.players;
    }

    public void setPlayers(Collection<UUID> players) {
        this.players.clear();
        //noinspection ConstantConditions
        PlayerList playerList = this.getLevel().getServer().getPlayerList();
        if (ModList.get().isLoaded("minemention")) {
            for (UUID id : players) {
                MineMentionCompat.updateMentions(playerList.getPlayer(id));
            }
        }
        this.players.addAll(players);
        this.updateLastChanged();
    }

    public Set<TemplatesConfig.Spawn> getPossibleSpawns() {
        return Set.copyOf(this.possibleSpawns);
    }

    public Set<TemplatesConfig.Spawn> getDefaultPossibleSpawns() {
        return Set.copyOf(this.defaultPossibleSpawns);
    }

    public void setPossibleSpawns(Collection<TemplatesConfig.Spawn> spawns) {
        this.possibleSpawns.clear();
        this.defaultPossibleSpawns.clear();
        this.possibleSpawns.addAll(spawns);
        this.defaultPossibleSpawns.addAll(spawns);
        this.updateLastChanged();
    }

    public void addPossibleSpawn(TemplatesConfig.Spawn spawn) {
        this.possibleSpawns.add(spawn);
        this.updateLastChanged();
    }

    public void addPossibleSpawn(BlockPos pos, WorldUtil.SpawnDirection direction) {
        this.addPossibleSpawn(new TemplatesConfig.Spawn(pos, direction));
    }

    public boolean removePossibleSpawn(BlockPos pos) {
        if (this.possibleSpawns.size() <= 1) {
            return false;
        }

        for (TemplatesConfig.Spawn possibleSpawn : this.possibleSpawns) {
            if (possibleSpawn.pos().equals(pos)) {
                boolean remove = this.possibleSpawns.remove(possibleSpawn);
                this.updateLastChanged();

                return remove;
            }
        }

        return false;
    }

    public boolean allowsVisits() {
        return this.allowVisits;
    }

    public boolean toggleAllowVisits() {
        this.allowVisits = !this.allowVisits;
        this.updateLastChanged();
        return this.allowVisits;
    }

    public void setAllowVisit(boolean enabled) {
        if (this.allowVisits != enabled) {
            this.allowVisits = enabled;
            this.updateLastChanged();
        }
    }

    public boolean addPlayer(UUID player) {
        boolean added = this.players.add(player);
        if (added) {
            if (ModList.get().isLoaded("minemention")) {
                //noinspection ConstantConditions
                MineMentionCompat.updateMentions(this.getLevel().getServer().getPlayerList().getPlayer(player));
            }
            if (!this.isSpawn()) {
                this.data.getSpawn().removePlayer(player);
            }
            this.data.getOrCreateMetaInfo(player).setTeamId(this.teamId);
            this.updateLastChanged();
        }
        return added;
    }

    public boolean addPlayer(Player player) {
        return this.addPlayer(player.getGameProfile().getId());
    }

    public boolean addPlayers(Collection<UUID> players) {
        boolean added = this.players.addAll(players);
        if (added) {
            if (ModList.get().isLoaded("minemention")) {
                for (UUID id : players) {
                    //noinspection ConstantConditions
                    MineMentionCompat.updateMentions(this.getLevel().getServer().getPlayerList().getPlayer(id));
                    this.data.getOrCreateMetaInfo(id).setTeamId(this.teamId);
                }
            }
            this.updateLastChanged();
        }

        return added;
    }

    public boolean removePlayer(Player player) {
        return this.removePlayer(player.getGameProfile().getId());
    }

    public boolean removePlayer(UUID player) {
        boolean removed = this.players.remove(player);
        if (ModList.get().isLoaded("minemention")) {
            //noinspection ConstantConditions
            MineMentionCompat.updateMentions(this.getLevel().getServer().getPlayerList().getPlayer(player));
        }
        if (removed) {
            this.data.getOrCreateMetaInfo(player).addPreviousTeamId(this.teamId);
            this.lastChanged = System.currentTimeMillis();
        }
        this.data.setDirty();

        return removed;
    }

    public void removePlayers(Collection<UUID> players) {
        for (UUID id : players) {
            boolean removed = this.players.remove(id);
            if (ModList.get().isLoaded("minemention")) {
                //noinspection ConstantConditions
                MineMentionCompat.updateMentions(this.getLevel().getServer().getPlayerList().getPlayer(id));
            }
            if (removed) {
                this.data.getOrCreateMetaInfo(id).addPreviousTeamId(this.teamId);
            }
        }
        this.updateLastChanged();
    }

    public void removeAllPlayers() {
        this.removePlayers(this.players);
    }

    public boolean hasPlayer(UUID player) {
        return this.players.contains(player);
    }

    public boolean hasPlayer(Player player) {
        return this.hasPlayer(player.getGameProfile().getId());
    }

    public boolean isEmpty() {
        return this.players.isEmpty();
    }

    public boolean allowsJoinRequests() {
        return this.allowJoinRequests;
    }

    public boolean toggleAllowJoinRequest() {
        this.allowJoinRequests = !this.allowJoinRequests;
        this.updateLastChanged();
        return this.allowJoinRequests;
    }

    public void setAllowJoinRequest(boolean enabled) {
        if (this.allowJoinRequests != enabled) {
            this.allowJoinRequests = enabled;
            this.updateLastChanged();
        }
    }

    public Set<UUID> getJoinRequests() {
        return this.joinRequests;
    }

    public void addJoinRequest(Player player) {
        this.addJoinRequest(player.getGameProfile().getId());
    }

    public void addJoinRequest(UUID id) {
        this.joinRequests.add(id);
        this.data.setDirty();
    }

    public void removeJoinRequest(Player player) {
        this.removeJoinRequest(player.getGameProfile().getId());
    }

    public void removeJoinRequest(UUID id) {
        this.joinRequests.remove(id);
        this.data.setDirty();
    }

    public void resetJoinRequests() {
        this.joinRequests.clear();
        this.data.setDirty();
    }

    public void addSpread(String spreadName, BlockPos pos, BlockPos size) {
        this.addSpread(new PlacedSpread(spreadName, pos, size));
    }

    public void addSpread(PlacedSpread placedSpread) {
        this.placedSpreads.computeIfAbsent(placedSpread.name(), s -> new HashSet<>()).add(placedSpread);
        this.data.setDirty();
    }

    public Map<String, Set<PlacedSpread>> getPlacedSpreads() {
        return this.placedSpreads;
    }

    public Set<PlacedSpread> getPlacedSpreads(String spreadName) {
        return this.placedSpreads.containsKey(spreadName) ? this.placedSpreads.get(spreadName) : Set.of();
    }

    public Set<String> getAllSpreadNames() {
        return this.placedSpreads.keySet();
    }

    public void sendJoinRequest(Player requestingPlayer) {
        this.addJoinRequest(requestingPlayer.getGameProfile().getId());
        MutableComponent component = SkyComponents.EVENT_JOIN_REQUEST0.apply(requestingPlayer.getDisplayName());
        component.append(Component.literal("/skyblock team accept " + requestingPlayer.getDisplayName().getString()).setStyle(Style.EMPTY
                .withHoverEvent(InviteCommand.COPY_TEXT)
                .withClickEvent(new ClickEvent(ClickEvent.Action.SUGGEST_COMMAND, "/skyblock team accept " + requestingPlayer.getDisplayName().getString()))
                .applyFormats(ChatFormatting.UNDERLINE, ChatFormatting.GOLD)));
        component.append(SkyComponents.EVENT_JOIN_REQUEST1);
        this.broadcast(component, Style.EMPTY.applyFormat(ChatFormatting.GOLD));
    }

    public long getCreatedAt() {
        return this.createdAt;
    }

    public long getLastChanged() {
        return this.lastChanged;
    }

    public void updateLastChanged() {
        this.lastChanged = System.currentTimeMillis();
        this.data.setDirty();
    }

    @Nullable
    public ServerLevel getLevel() {
        return this.data.getLevel();
    }

    public void broadcast(MutableComponent msg, Style style) {
        if (this.getLevel() == null || this.getLevel().isClientSide) {
            return;
        }

        PlayerList playerList = this.getLevel().getServer().getPlayerList();
        this.players.forEach(uuid -> {
            ServerPlayer player = playerList.getPlayer(uuid);
            if (player != null) {
                MutableComponent component = Component.literal("[" + this.name + "] ").setStyle(Style.EMPTY);
                player.sendSystemMessage(component.append(msg.withStyle(style)));
            }
        });
    }

    @Nonnull
    public CompoundTag serializeNBT() {
        CompoundTag nbt = new CompoundTag();

        nbt.putUUID(TEAM_ID, this.teamId);
        nbt.put(ISLAND, this.island.toTag());
        nbt.putString(NAME, this.name != null ? this.name : "");
        nbt.putBoolean(VISITS, this.allowVisits);
        nbt.putBoolean(ALLOW_JOIN_REQUESTS, this.allowJoinRequests);
        nbt.putLong(CREATED_AT, this.createdAt);
        nbt.putLong(LAST_CHANGED, this.lastChanged);

        ListTag players = new ListTag();
        for (UUID player : this.players) {
            CompoundTag playerTag = new CompoundTag();
            playerTag.putUUID(PLAYER, player);
            players.add(playerTag);
        }

        ListTag spawns = new ListTag();
        for (TemplatesConfig.Spawn spawn : this.possibleSpawns) {
            CompoundTag posTag = WorldUtil.blockPosToTag(spawn.pos());
            posTag.putString(DIRECTION, spawn.direction().name());
            spawns.add(posTag);
        }

        ListTag defaultSpawns = new ListTag();
        for (TemplatesConfig.Spawn spawn : this.defaultPossibleSpawns) {
            CompoundTag posTag = WorldUtil.blockPosToTag(spawn.pos());
            posTag.putString(DIRECTION, spawn.direction().name());
            defaultSpawns.add(posTag);
        }

        ListTag joinRequests = new ListTag();
        for (UUID id : this.joinRequests) {
            CompoundTag idTag = new CompoundTag();
            idTag.putUUID(ID, id);
            joinRequests.add(idTag);
        }

        CompoundTag placedSpreads = new CompoundTag();
        for (Map.Entry<String, Set<PlacedSpread>> entry : this.placedSpreads.entrySet()) {
            ListTag namedSpreads = new ListTag();
            for (PlacedSpread placedSpread : entry.getValue()) {
                CompoundTag tag = new CompoundTag();
                tag.putString(NAME, placedSpread.name());
                tag.put(POS, WorldUtil.blockPosToTag(placedSpread.pos()));
                tag.put(SIZE, WorldUtil.blockPosToTag(placedSpread.size()));
                namedSpreads.add(tag);
            }
            placedSpreads.put(entry.getKey(), namedSpreads);
        }
        nbt.put(PLACED_SPREADS, placedSpreads);

        nbt.put(PLAYERS, players);
        nbt.put(SPAWNS, spawns);
        nbt.put(DEFAULT_SPAWNS, defaultSpawns);
        nbt.put(JOIN_REQUESTS, joinRequests);
        return nbt;
    }

    public void deserializeNBT(CompoundTag nbt) {
        this.teamId = nbt.getUUID(TEAM_ID);
        this.island = IslandPos.fromTag(nbt.getCompound(ISLAND));
        this.name = nbt.getString(NAME);
        this.allowVisits = nbt.getBoolean(VISITS);
        this.allowJoinRequests = nbt.getBoolean(ALLOW_JOIN_REQUESTS);
        this.createdAt = nbt.getLong(CREATED_AT);
        this.lastChanged = nbt.getLong(LAST_CHANGED);

        ListTag players = nbt.getList(PLAYERS, Tag.TAG_COMPOUND);
        this.players.clear();
        for (Tag player : players) {
            this.players.add(((CompoundTag) player).getUUID(PLAYER));
        }

        ListTag spawns = nbt.getList(SPAWNS, Tag.TAG_COMPOUND);
        this.possibleSpawns.clear();
        for (Tag tag : spawns) {
            CompoundTag posTag = (CompoundTag) tag;
            BlockPos pos = WorldUtil.blockPosFromTag(posTag);
            WorldUtil.SpawnDirection direction = WorldUtil.SpawnDirection.valueOf(posTag.getString(DIRECTION));
            this.possibleSpawns.add(new TemplatesConfig.Spawn(pos, direction));
        }

        ListTag defaultSpawns = nbt.getList(DEFAULT_SPAWNS, Tag.TAG_COMPOUND);
        this.defaultPossibleSpawns.clear();
        for (Tag tag : defaultSpawns) {
            CompoundTag posTag = (CompoundTag) tag;
            BlockPos pos = WorldUtil.blockPosFromTag(posTag);
            WorldUtil.SpawnDirection direction = WorldUtil.SpawnDirection.valueOf(posTag.getString(DIRECTION));
            this.defaultPossibleSpawns.add(new TemplatesConfig.Spawn(pos, direction));
        }

        ListTag joinRequests = nbt.getList(JOIN_REQUESTS, Tag.TAG_COMPOUND);
        this.joinRequests.clear();
        for (Tag id : joinRequests) {
            this.joinRequests.add(((CompoundTag) id).getUUID(ID));
        }

        CompoundTag placedSpreads = nbt.getCompound(PLACED_SPREADS);
        this.placedSpreads.clear();
        for (String key : placedSpreads.getAllKeys()) {
            ListTag list = placedSpreads.getList(key, Tag.TAG_COMPOUND);
            Set<PlacedSpread> namedSpreads = new HashSet<>();
            for (Tag tag : list) {
                CompoundTag ctag = ((CompoundTag) tag);
                String name = ctag.getString(NAME);
                BlockPos pos = WorldUtil.blockPosFromTag(ctag.getCompound(POS));
                BlockPos size = WorldUtil.blockPosFromTag(ctag.getCompound(SIZE));

                PlacedSpread placedSpread = new PlacedSpread(name, pos, size);
                namedSpreads.add(placedSpread);
            }
            this.placedSpreads.put(key, namedSpreads);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        } else if (!(o instanceof Team)) {
            return false;
        }

        Team team = (Team) o;
        return this.name.equals(team.name) && this.island.equals(team.island);
    }

    @Override
    public int hashCode() {
        int result = this.name.hashCode();
        result = 31 * result * this.island.hashCode();
        return result;
    }

    public record PlacedSpread(String name, BlockPos pos, BlockPos size) {}
}
