package de.melanx.skyblockbuilder.config.mapper;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import de.melanx.skyblockbuilder.SkyblockBuilder;
import de.melanx.skyblockbuilder.config.common.TemplatesConfig;
import de.melanx.skyblockbuilder.config.values.providers.SpawnsProvider;
import de.melanx.skyblockbuilder.config.values.providers.SpreadsProvider;
import de.melanx.skyblockbuilder.config.values.providers.SurroundingBlocksProvider;
import de.melanx.skyblockbuilder.template.TemplateInfo;
import net.minecraft.core.BlockPos;
import org.moddingx.libx.annotation.config.RegisterMapper;
import org.moddingx.libx.config.gui.ConfigEditor;
import org.moddingx.libx.config.mapper.ValueMapper;
import org.moddingx.libx.config.validator.ValidatorInfo;
import org.moddingx.libx.impl.config.ModMappers;
import org.moddingx.libx.impl.config.mappers.special.RecordValueMapper;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;

@RegisterMapper
public class TemplateInfoMapper implements ValueMapper<TemplateInfo, JsonObject> {

    @Override
    public Class<TemplateInfo> type() {
        return TemplateInfo.class;
    }

    @Override
    public Class<JsonObject> element() {
        return JsonObject.class;
    }

    @Override
    public TemplateInfo fromJson(JsonObject json) {
        String name = json.get("name").getAsString();
        String file = json.get("file").getAsString();
        SpawnsProvider spawns = SpawnsProvider.fromJson(json.get("spawns"));

        String desc = "";
        if (json.has("desc")) {
            desc = json.get("desc").getAsString();
        }

        BlockPos offset = new BlockPos(TemplatesConfig.defaultOffset, 0, TemplatesConfig.defaultOffset);
        if (json.has("offset")) {
            JsonArray offsetArray = json.get("offset").getAsJsonArray();
            offset = new BlockPos(offsetArray.get(0).getAsInt() + TemplatesConfig.defaultOffset, offsetArray.get(1).getAsInt(), offsetArray.get(2).getAsInt() + TemplatesConfig.defaultOffset);
        } else if (TemplatesConfig.defaultOffset != 0) {
            offset = new BlockPos(TemplatesConfig.defaultOffset, 0, TemplatesConfig.defaultOffset);
        }

        SurroundingBlocksProvider surroundingBlocks = SurroundingBlocksProvider.EMPTY;
        if (json.has("surroundingBlocks")) {
            surroundingBlocks = SurroundingBlocksProvider.fromJson(json.get("surroundingBlocks"));
        }

        SpreadsProvider spreads = SpreadsProvider.EMPTY;
        if (json.has("spreads")) {
            spreads = SpreadsProvider.fromJson(json.get("spreads"));
        }

        boolean allowPaletteSelection = true;
        if (json.has("allowPaletteSelection")) {
            allowPaletteSelection = json.get("allowPaletteSelection").getAsBoolean();
        }

        return new TemplateInfo(name, desc, file, spawns, offset, surroundingBlocks, spreads, allowPaletteSelection);
    }

    @Override
    public JsonObject toJson(TemplateInfo templateInfo) {
        JsonObject json = new JsonObject();
        json.addProperty("name", templateInfo.name());

        if (!templateInfo.desc().isBlank()) {
            json.addProperty("desc", templateInfo.desc());
        }

        json.addProperty("file", templateInfo.file());
        json.add("spawns", templateInfo.spawns().toJson());

        if (templateInfo.offset().getX() != TemplatesConfig.defaultOffset || templateInfo.offset().getZ() != TemplatesConfig.defaultOffset) {
            JsonArray offsetArray = new JsonArray();
            offsetArray.add(templateInfo.offset().getX());
            offsetArray.add(templateInfo.offset().getY());
            offsetArray.add(templateInfo.offset().getZ());
            json.add("offset", offsetArray);
        }

        if (templateInfo.surroundingBlocks() != null && !templateInfo.surroundingBlocks().isEmpty()) {
            json.add("surroundingBlocks", templateInfo.surroundingBlocks().toJson());
        }

        if (templateInfo.spreads() != null && !templateInfo.spreads().isEmpty()) {
            json.add("spreads", templateInfo.spreads().toJson());
        }

        if (!templateInfo.allowPaletteSelection()) {
            json.addProperty("allowPaletteSelection", false);
        }

        return json;
    }


    @Override
    public ConfigEditor<TemplateInfo> createEditor(ValidatorInfo<?> validator) {
        try {
            Method method = ModMappers.class.getDeclaredMethod("getMapper", Type.class);
            method.setAccessible(true);

            ModMappers modMappers = ModMappers.get(SkyblockBuilder.getInstance().modid);
            return new RecordValueMapper<>(SkyblockBuilder.getInstance().modid, TemplateInfo.class, type -> {
                try {
                    return (ValueMapper<?, ?>) method.invoke(modMappers, type);
                } catch (IllegalAccessException | InvocationTargetException e) {
                    throw new RuntimeException(e);
                }
            }).createEditor(validator);
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        }
    }
}
