package de.melanx.skyblockbuilder.world.chunkgenerators;

import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import de.melanx.skyblockbuilder.config.common.StructuresConfig;
import de.melanx.skyblockbuilder.config.common.WorldConfig;
import de.melanx.skyblockbuilder.world.flat.FlatLayerConfig;
import de.melanx.skyblockbuilder.world.flat.FlatLayers;
import it.unimi.dsi.fastutil.ints.IntArraySet;
import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.fastutil.objects.ObjectArraySet;
import net.minecraft.CrashReport;
import net.minecraft.ReportedException;
import net.minecraft.core.*;
import net.minecraft.core.registries.Registries;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.WorldGenRegion;
import net.minecraft.world.level.*;
import net.minecraft.world.level.biome.*;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.chunk.*;
import net.minecraft.world.level.levelgen.*;
import net.minecraft.world.level.levelgen.blending.Blender;
import net.minecraft.world.level.levelgen.carver.CarvingContext;
import net.minecraft.world.level.levelgen.carver.ConfiguredWorldCarver;
import net.minecraft.world.level.levelgen.feature.ConfiguredFeature;
import net.minecraft.world.level.levelgen.placement.PlacedFeature;
import net.minecraft.world.level.levelgen.structure.Structure;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public class SkyblockNoiseBasedChunkGenerator extends NoiseBasedChunkGenerator {

    // [VanillaCopy] overworld chunk generator codec
    public static final MapCodec<SkyblockNoiseBasedChunkGenerator> CODEC = RecordCodecBuilder.mapCodec(
            (instance) -> instance.group(
                    BiomeSource.CODEC.fieldOf("biome_source").forGetter(generator -> generator.biomeSource),
                    NoiseGeneratorSettings.CODEC.fieldOf("settings").forGetter(generator -> generator.generatorSettings),
                    Level.RESOURCE_KEY_CODEC.fieldOf("dimension").forGetter(generator -> generator.dimension),
                    FlatLayers.CODEC.optionalFieldOf("layers", FlatLayers.EMPTY).forGetter(generator -> generator.flatLayers)
            ).apply(instance, instance.stable(SkyblockNoiseBasedChunkGenerator::new)));

    public final Holder<NoiseGeneratorSettings> generatorSettings;
    public final ResourceKey<Level> dimension;
    protected final NoiseBasedChunkGenerator parent;
    protected final FlatLayers flatLayers;
    private final int layerHeight;

    public SkyblockNoiseBasedChunkGenerator(BiomeSource biomeSource, Holder<NoiseGeneratorSettings> generatorSettings, ResourceKey<Level> dimension, FlatLayers flatLayers) {
        super(biomeSource, generatorSettings);
        this.generatorSettings = generatorSettings;
        this.parent = new NoiseBasedChunkGenerator(biomeSource, generatorSettings);
        this.dimension = dimension;
        this.flatLayers = flatLayers;
        this.layerHeight = this.flatLayers.totalHeight();
    }

    @Nonnull
    @Override
    protected MapCodec<? extends ChunkGenerator> codec() {
        return SkyblockNoiseBasedChunkGenerator.CODEC;
    }

    @Override
    public int getSeaLevel() {
        return WorldConfig.seaHeight;
    }

    @Override
    public void buildSurface(@Nonnull WorldGenRegion level, @Nonnull StructureManager structureManager, @Nonnull RandomState randomState, @Nonnull ChunkAccess chunk) {
        if (!this.flatLayers.isEmpty()) {
            ChunkPos cp = chunk.getPos();
            int xs = cp.getMinBlockX();
            int zs = cp.getMinBlockZ();
            int xe = cp.getMaxBlockX();
            int ze = cp.getMaxBlockZ();
            int y = level.getMinBuildHeight();
            BlockPos.MutableBlockPos pos = new BlockPos.MutableBlockPos();
            for (FlatLayerConfig info : this.flatLayers.layers()) {
                BlockState state = info.getBlockState();

                for (int i = 0; i < info.getHeight(); i++) {
                    for (int x = xs; x <= xe; x++) {
                        for (int z = zs; z <= ze; z++) {
                            if (info.hasExtra()) {
                                if (!info.checkChance(level.getRandom())) {
                                    state = info.getBlockState();
                                } else {
                                    Optional<FlatLayerConfig.WeightedBlockEntry> maybeBlock = info.getExtraBlocks().getRandom(level.getRandom());
                                    if (maybeBlock.isPresent()) {
                                        state = maybeBlock.get().block().defaultBlockState();
                                    }
                                }
                            }

                            pos.setX(x);
                            pos.setY(y);
                            pos.setZ(z);
                            chunk.setBlockState(pos, state, false);
                        }
                    }
                    y++;
                }
            }
        }
    }

    @Nonnull
    @Override
    public CompletableFuture<ChunkAccess> fillFromNoise(@Nonnull Blender blender, @Nonnull RandomState randomState, @Nonnull StructureManager structureManager, @Nonnull ChunkAccess chunk) {
        return CompletableFuture.completedFuture(chunk);
    }

    @Nullable
    @Override
    public Pair<BlockPos, Holder<Structure>> findNearestMapStructure(@Nonnull ServerLevel level, @Nonnull HolderSet<Structure> structureHolderSet, @Nonnull BlockPos pos, int searchRadius, boolean skipKnownStructures) {
        List<Holder<Structure>> holders = structureHolderSet.stream().filter(holder -> holder.unwrapKey().isPresent() && StructuresConfig.structuresToGenerate.test(holder.unwrapKey().get().location())).toList();

        if (holders.isEmpty()) {
            return null;
        }

        for (Holder<Structure> holder : holders) {
            if (holder.unwrapKey().isPresent()) {
                HolderSet.Direct<Structure> modifiedStructureHolderSet = HolderSet.direct(holders);
                return super.findNearestMapStructure(level, modifiedStructureHolderSet, pos, searchRadius, skipKnownStructures);
            }
        }

        return null;
    }

    @Override
    public int getBaseHeight(int x, int z, @Nonnull Heightmap.Types heightmapType, @Nonnull LevelHeightAccessor level, @Nonnull RandomState randomState) {
        if (WorldConfig.surface) {
            return level.getMinBuildHeight() + this.layerHeight;
        }

        return this.parent.getBaseHeight(x, z, heightmapType, level, randomState);
    }

    @Override
    public void applyCarvers(@Nonnull WorldGenRegion level, long seed, @Nonnull RandomState random, @Nonnull BiomeManager biomeManager, @Nonnull StructureManager structureManager, @Nonnull ChunkAccess chunk, @Nonnull GenerationStep.Carving step) {
        if (this.flatLayers.isEmpty()) {
            return;
        }

        NoiseChunk noiseChunk = chunk.getOrCreateNoiseChunk((otherChunk) -> this.createNoiseChunk(otherChunk, structureManager, Blender.of(level), random));

        ChunkPos chunkPos = chunk.getPos();
        Aquifer aquifer = noiseChunk.aquifer();
        CarvingMask carvingMask = ((ProtoChunk) chunk).getOrCreateCarvingMask(step);
        WorldgenRandom worldGenRandom = new WorldgenRandom(new LegacyRandomSource(RandomSupport.generateUniqueSeed()));
        BiomeManager differentBiomeManager = biomeManager.withDifferentSource((x, y, z) -> this.biomeSource.getNoiseBiome(x, y, z, random.sampler()));
        CarvingContext carvingContext = new CarvingContext(this, level.registryAccess(), chunk.getHeightAccessorForGeneration(), noiseChunk, random, this.settings.value().surfaceRule());

        int i = 8;
        for (int j = -i; j <= i; ++j) {
            for (int k = -i; k <= i; ++k) {
                ChunkPos tempChunkPos = new ChunkPos(chunkPos.x + j, chunkPos.z + k);
                //noinspection deprecation
                BiomeGenerationSettings biomeGenerationSettings = level.getChunk(tempChunkPos.x, tempChunkPos.z).carverBiome(() -> {
                    //noinspection deprecation
                    return this.getBiomeGenerationSettings(this.biomeSource.getNoiseBiome(QuartPos.fromBlock(tempChunkPos.getMinBlockX()), 0, QuartPos.fromBlock(tempChunkPos.getMinBlockZ()), random.sampler()));
                });

                int l = 0;
                for (Holder<ConfiguredWorldCarver<?>> holder : biomeGenerationSettings.getCarvers(step)) {
                    // my change
                    if (holder.unwrapKey().isPresent() && !WorldConfig.carvers.get(this.dimension.location().toString()).test(holder.unwrapKey().get().location())) {
                        continue;
                    }

                    ConfiguredWorldCarver<?> configuredCarver = holder.value();
                    worldGenRandom.setLargeFeatureSeed(seed + (long) l, tempChunkPos.x, tempChunkPos.z);
                    if (configuredCarver.isStartChunk(worldGenRandom)) {
                        configuredCarver.carve(carvingContext, chunk, differentBiomeManager::getBiome, worldGenRandom, aquifer, tempChunkPos, carvingMask);
                    }

                    l++;
                }
            }
        }
    }

    @Nonnull
    @Override
    public NoiseColumn getBaseColumn(int posX, int posZ, @Nonnull LevelHeightAccessor level, @Nonnull RandomState randomState) {
        return new NoiseColumn(0, new BlockState[0]);
    }

    // [Vanilla copy] to ignore some features
    @Override
    public void applyBiomeDecoration(@Nonnull WorldGenLevel level, @Nonnull ChunkAccess chunk, @Nonnull StructureManager structureManager) {
        ChunkPos chunkPos = chunk.getPos();
        SectionPos sectionPos = SectionPos.of(chunkPos, level.getMinSection());
        BlockPos blockpos = sectionPos.origin();

        Registry<Structure> structureRegistry = level.registryAccess().registryOrThrow(Registries.STRUCTURE);
        Map<Integer, List<Structure>> map = structureRegistry.stream().collect(Collectors.groupingBy(structure -> structure.step().ordinal()));
        List<FeatureSorter.StepFeatureData> stepFeatureDataList = this.featuresPerStep.get();
        WorldgenRandom worldgenRandom = new WorldgenRandom(new XoroshiroRandomSource(RandomSupport.generateUniqueSeed()));
        long decorationSeed = worldgenRandom.setDecorationSeed(level.getSeed(), blockpos.getX(), blockpos.getZ());

        Set<Holder<Biome>> possibleBiomes = new ObjectArraySet<>();
        ChunkPos.rangeClosed(sectionPos.chunk(), 1).forEach((pos) -> {
            ChunkAccess chunkaccess = level.getChunk(pos.x, pos.z);
            for (LevelChunkSection chunkSection : chunkaccess.getSections()) {
                chunkSection.getBiomes().getAll(possibleBiomes::add);
            }
        });
        possibleBiomes.retainAll(this.biomeSource.possibleBiomes());

        int dataSize = stepFeatureDataList.size();

        try {
            Registry<PlacedFeature> placedFeatureRegistry = level.registryAccess().registryOrThrow(Registries.PLACED_FEATURE);
            int maxDecorations = Math.max(GenerationStep.Decoration.values().length, dataSize);

            for (int i = 0; i < maxDecorations; ++i) {
                int index = 0;
                if (structureManager.shouldGenerateStructures()) {
                    for (Structure structure : map.getOrDefault(i, Collections.emptyList())) {
                        ResourceLocation location = level.registryAccess().registryOrThrow(Registries.STRUCTURE).getKey(structure);
                        if (!StructuresConfig.structuresToGenerate.test(location)) {
                            continue;
                        }

                        worldgenRandom.setFeatureSeed(decorationSeed, index, i);
                        Supplier<String> currentlyGenerating = () -> structureRegistry.getResourceKey(structure).map(Object::toString).orElseGet(structure::toString);

                        try {
                            level.setCurrentlyGenerating(currentlyGenerating);
                            structureManager.startsForStructure(sectionPos, structure).forEach(
                                    structureStart -> structureStart.placeInChunk(level, structureManager, this, worldgenRandom, getWritableArea(chunk), chunkPos));
                        } catch (Exception e) {
                            CrashReport report = CrashReport.forThrowable(e, "Feature placement");
                            report.addCategory("Feature").setDetail("Description", currentlyGenerating::get);
                            throw new ReportedException(report);
                        }

                        index++;
                    }
                }

                if (i < dataSize) {
                    IntSet mapping = new IntArraySet();

                    for (Holder<Biome> holder : possibleBiomes) {
                        List<HolderSet<PlacedFeature>> holderSets = this.generationSettingsGetter.apply(holder).features();
                        if (i < holderSets.size()) {
                            HolderSet<PlacedFeature> featureHolderSet = holderSets.get(i);
                            FeatureSorter.StepFeatureData stepFeatureData = stepFeatureDataList.get(i);
                            featureHolderSet.stream().map(Holder::value).forEach(
                                    placedFeature -> mapping.add(stepFeatureData.indexMapping().applyAsInt(placedFeature)));
                        }
                    }

                    int mappingSize = mapping.size();
                    int[] array = mapping.toIntArray();
                    Arrays.sort(array);
                    FeatureSorter.StepFeatureData stepFeatureData = stepFeatureDataList.get(i);

                    for (int j = 0; j < mappingSize; ++j) {
                        int featureIndex = array[j];
                        PlacedFeature placedfeature = stepFeatureData.features().get(featureIndex);
                        // The only reason why I needed to copy the code - checking if it should be placed
                        Optional<ResourceKey<ConfiguredFeature<?, ?>>> optionalResourceKey = placedfeature.feature().unwrapKey();
                        if (optionalResourceKey.isPresent() && !StructuresConfig.featuresToGenerate.test(optionalResourceKey.get().location())) {
                            continue;
                        }

                        Supplier<String> currentlyGenerating = () -> placedFeatureRegistry.getResourceKey(placedfeature).map(Object::toString).orElseGet(placedfeature::toString);
                        worldgenRandom.setFeatureSeed(decorationSeed, featureIndex, i);

                        try {
                            level.setCurrentlyGenerating(currentlyGenerating);
                            placedfeature.placeWithBiomeCheck(level, this, worldgenRandom, blockpos);
                        } catch (Exception e) {
                            CrashReport report = CrashReport.forThrowable(e, "Feature placement");
                            report.addCategory("Feature").setDetail("Description", currentlyGenerating::get);
                            throw new ReportedException(report);
                        }
                    }
                }
            }

            level.setCurrentlyGenerating(null);
        } catch (Exception e) {
            CrashReport report = CrashReport.forThrowable(e, "Biome decoration");
            report.addCategory("Generation").setDetail("CenterX", chunkPos.x).setDetail("CenterZ", chunkPos.z).setDetail("Seed", decorationSeed);
            throw new ReportedException(report);
        }
    }

    public ResourceKey<Level> getDimension() {
        return this.dimension;
    }

    public FlatLayers getFlatLayers() {
        return this.flatLayers;
    }
}
