package de.melanx.skyguis.client.screen;

import de.melanx.skyblockbuilder.client.SizeableCheckbox;
import de.melanx.skyblockbuilder.template.ConfiguredTemplate;
import de.melanx.skyblockbuilder.template.TemplateLoader;
import de.melanx.skyblockbuilder.template.TemplateRenderer;
import de.melanx.skyblockbuilder.util.NameGenerator;
import de.melanx.skyguis.SkyGUIs;
import de.melanx.skyguis.util.ComponentBuilder;
import de.melanx.skyguis.util.TextHelper;
import net.minecraft.ChatFormatting;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.components.Button;
import net.minecraft.client.gui.components.EditBox;
import net.minecraft.client.gui.components.Tooltip;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;

import javax.annotation.Nonnull;
import java.awt.Color;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

public class CreateTeamScreen extends BaseScreen {

    private static final Component NAME_COMPONENT = ComponentBuilder.text("name");
    private static final Component TEMPLATE_COMPONENT = ComponentBuilder.raw("template");
    private static final Component SETTINGS_COMPONENT = ComponentBuilder.text("settings");
    private static final Component CREATE = ComponentBuilder.button("create");
    private static final Component ABORT = ComponentBuilder.button("abort");
    private static final Component TITLE = ComponentBuilder.title("create_team");
    private static final Component ALLOW_VISITS = ComponentBuilder.text("allow_visits");
    private static final Component ALLOW_REQUESTS = ComponentBuilder.text("allow_requests");

    private transient final Map<String, TemplateRenderer> structureCache = new HashMap<>();
    private final List<String> templates;
    private String currTemplate;
    private EditBox name;
    private int currIndex = 0;
    private boolean enableTooltip;
    private Button templateButton;
    private SizeableCheckbox allowVisits;
    private SizeableCheckbox allowJoinRequests;

    public CreateTeamScreen() {
        super(TITLE, 200, 147);
        this.templates = TemplateLoader.getTemplateNames();
    }

    public static void open() {
        Minecraft.getInstance().setScreen(new CreateTeamScreen());
    }

    @Override
    protected void init() {
        this.structureCache.clear();
        this.name = new EditBox(this.font, this.x(66), this.y(30), 120, 20, Component.empty());
        this.name.setMaxLength(Short.MAX_VALUE);
        this.name.setValue(this.name.getValue());
        this.name.setMaxLength(64);
        this.addRenderableWidget(this.name);

        if (this.templates.isEmpty()) {
            Minecraft.getInstance().setScreen(null);
            //noinspection ConstantConditions
            Minecraft.getInstance().player.sendSystemMessage(ComponentBuilder.text("empty_templates").withStyle(ChatFormatting.BOLD, ChatFormatting.RED));
            return;
        }

        Component shortened = this.setCurrentTemplateAndGetShortenedName();
        this.templateButton = Button.builder(shortened, button -> {
                    this.currIndex++;
                    if (this.currIndex >= this.templates.size()) {
                        this.currIndex = 0;
                    }

                    Component s = this.setCurrentTemplateAndGetShortenedName();
                    button.setMessage(s);
                    this.updateTemplateButton();
                })
                .bounds(this.x(65), this.y(60), 122, 20)
                .build();

        this.allowVisits = new SizeableCheckbox(this.x(65), this.y(85), 10, false);
        this.allowJoinRequests = new SizeableCheckbox(this.x(65), this.y(100), 10, false);

        this.addRenderableWidget(this.allowVisits);
        this.addRenderableWidget(this.allowJoinRequests);

        this.updateTemplateButton();
        if (this.templates.size() == 1) {
            this.templateButton.active = false;
        }
        this.currTemplate = this.templates.get(this.currIndex);
        this.addRenderableWidget(this.templateButton);

        this.addRenderableWidget(Button.builder(CREATE, button -> {
            if (this.name.getValue().isBlank()) {
                this.name.setFocused(true);
                this.name.setValue(NameGenerator.randomName(new Random()));
            } else {
                SkyGUIs.getNetwork().handleCreateTeam(this.name.getValue().strip(), this.currTemplate, this.allowVisits.selected, this.allowJoinRequests.selected);
            }
        }).bounds(this.x(27), this.y(116), 60, 20).build());
        this.addRenderableWidget(Button.builder(ABORT, button -> this.onClose()).bounds(this.x(106), this.y(116), 60, 20).build());
    }

    private Component setCurrentTemplateAndGetShortenedName() {
        String orig = this.templates.get(this.currIndex);
        ConfiguredTemplate configuredTemplate = TemplateLoader.getConfiguredTemplate(orig);
        if (configuredTemplate == null) {
            throw new IllegalStateException("Templates not synced between client and server: " + orig);
        }

        Component nameComponent = configuredTemplate.getNameComponent();
        String s = TextHelper.shorten(this.font, nameComponent.getString(), 110);
        String desc = configuredTemplate.getDescriptionComponent().getString();
        this.enableTooltip = !s.equals(nameComponent.getString()) || !desc.isBlank();
        this.currTemplate = orig;

        return Component.literal(s).setStyle(nameComponent.getStyle());
    }

    @Override
    public void renderBackground(@Nonnull GuiGraphics guiGraphics, int mouseX, int mouseY, float partialTick) {
        super.renderBackground(guiGraphics, mouseX, mouseY, partialTick);
        this.renderTitle(guiGraphics);
        guiGraphics.drawString(this.font, NAME_COMPONENT, this.x(10), this.y(37), Color.DARK_GRAY.getRGB(), false);
        guiGraphics.drawString(this.font, TEMPLATE_COMPONENT, this.x(10), this.y(67), Color.DARK_GRAY.getRGB(), false);
        guiGraphics.drawString(this.font, SETTINGS_COMPONENT, this.x(10), this.y(92), Color.DARK_GRAY.getRGB(), false);
        if (!this.structureCache.containsKey(this.currTemplate)) {
            SkyGUIs.getNetwork().requestTemplateFromServer(this.currTemplate);
            this.structureCache.put(this.currTemplate, null);
            return;
        }

        TemplateRenderer renderer = this.structureCache.get(this.currTemplate);
        if (renderer != null) {
            renderer.render(guiGraphics, this.x(0) / 2, this.centeredY(0));
        }

        float scale = 0.9f;
        guiGraphics.pose().pushPose();
        guiGraphics.pose().scale(scale, scale, scale);
        guiGraphics.drawString(this.font, ALLOW_VISITS, (int) (this.x(82) / scale), (int) (this.y(87) / scale), Color.DARK_GRAY.getRGB(), false);
        guiGraphics.drawString(this.font, ALLOW_REQUESTS, (int) (this.x(82) / scale), (int) (this.y(102) / scale), Color.DARK_GRAY.getRGB(), false);
        guiGraphics.pose().scale(1 / 0.8f, 1 / 0.8f, 1 / 0.8f);
        guiGraphics.pose().popPose();
    }

    public void addStructureToCache(String name, ConfiguredTemplate template) {
        this.structureCache.put(name, new TemplateRenderer(template.getTemplate(), Math.min((float) (this.x(0) * 0.9), this.height)));
    }

    public void updateTemplateButton() {
        if (this.enableTooltip) {
            ConfiguredTemplate configuredTemplate = TemplateLoader.getConfiguredTemplate(this.currTemplate);
            if (configuredTemplate == null) {
                throw new IllegalStateException("Template does not exist: " + this.currTemplate);
            }

            MutableComponent nameComponent = configuredTemplate.getNameComponent().copy();
            MutableComponent descComponent = configuredTemplate.getDescriptionComponent().copy().withStyle(ChatFormatting.ITALIC, ChatFormatting.GRAY);
            this.templateButton.setTooltip(Tooltip.create(nameComponent.append("\n").append(descComponent)));
        }
    }
}
