/*
 * Decompiled with CFR 0.152.
 */
package de.melanx.simplebackups;

import de.melanx.simplebackups.BackupChain;
import de.melanx.simplebackups.BackupChainManager;
import de.melanx.simplebackups.BackupData;
import de.melanx.simplebackups.SimpleBackups;
import de.melanx.simplebackups.StorageSize;
import de.melanx.simplebackups.compat.CherishedWorldsCompat;
import de.melanx.simplebackups.compat.Mc2DiscordCompat;
import de.melanx.simplebackups.config.BackupType;
import de.melanx.simplebackups.config.CommonConfig;
import de.melanx.simplebackups.config.ServerConfig;
import de.melanx.simplebackups.exception.NotEnoughDiskSpaceException;
import de.melanx.simplebackups.network.Pause;
import java.io.BufferedOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileStore;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.SignStyle;
import java.time.temporal.ChronoField;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import net.minecraft.ChatFormatting;
import net.minecraft.DefaultUncaughtExceptionHandler;
import net.minecraft.network.Connection;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.HoverEvent;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.network.chat.Style;
import net.minecraft.resources.Identifier;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.server.permissions.Permissions;
import net.minecraft.world.level.storage.LevelStorageSource;
import net.neoforged.fml.i18n.FMLTranslations;
import net.neoforged.neoforge.network.registration.NetworkRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackupThread
extends Thread {
    private static final DateTimeFormatter FORMATTER = new DateTimeFormatterBuilder().appendValue(ChronoField.YEAR, 4, 10, SignStyle.EXCEEDS_PAD).appendLiteral('-').appendValue(ChronoField.MONTH_OF_YEAR, 2).appendLiteral('-').appendValue(ChronoField.DAY_OF_MONTH, 2).appendLiteral('_').appendValue(ChronoField.HOUR_OF_DAY, 2).appendLiteral('-').appendValue(ChronoField.MINUTE_OF_HOUR, 2).appendLiteral('-').appendValue(ChronoField.SECOND_OF_MINUTE, 2).toFormatter();
    public static final Logger LOGGER = LoggerFactory.getLogger(BackupThread.class);
    public static final long BACKUP_BUFFER_SIZE = 0x8000000L;
    private final MinecraftServer server;
    private final boolean quiet;
    private final long lastSaved;
    private final boolean fullBackup;
    private final LevelStorageSource.LevelStorageAccess storageSource;
    private final Path backupPath;
    private final Set<Path> errors = new HashSet<Path>();
    private final BackupChainManager manager;
    private boolean forceFullBackup = false;

    private BackupThread(@Nonnull MinecraftServer server, boolean quiet, BackupData backupData) {
        this.server = server;
        this.storageSource = server.storageSource;
        this.quiet = quiet;
        if (backupData == null) {
            this.lastSaved = 0L;
            this.fullBackup = true;
        } else {
            long now = CommonConfig.useTickCounter() ? server.overworld().getGameTime() : System.currentTimeMillis();
            this.lastSaved = CommonConfig.backupType() == BackupType.INCREMENTAL ? backupData.getLastSaved() : backupData.getLastFullBackup();
            this.fullBackup = CommonConfig.backupType() == BackupType.FULL_BACKUPS || now - CommonConfig.getFullBackupTimer() > backupData.getLastFullBackup();
        }
        this.setName("SimpleBackups");
        this.setUncaughtExceptionHandler((Thread.UncaughtExceptionHandler)new DefaultUncaughtExceptionHandler(LOGGER));
        String levelId = this.storageSource.getLevelId();
        this.backupPath = CommonConfig.getOutputPath(levelId);
        this.manager = BackupChainManager.get(levelId);
    }

    public static boolean tryCreateBackup(MinecraftServer server) {
        BackupData backupData = BackupData.get(server);
        if (BackupThread.shouldRunBackup(server)) {
            BackupThread thread = new BackupThread(server, false, backupData);
            thread.start();
            long currentTime = CommonConfig.useTickCounter() ? server.overworld().getGameTime() : System.currentTimeMillis();
            backupData.updateSaveTime(currentTime);
            if (thread.createFullBackup()) {
                backupData.updateFullBackupTime(currentTime);
            }
            return true;
        }
        return false;
    }

    public static boolean shouldRunBackup(MinecraftServer server) {
        boolean arePlayersOnline;
        BackupData backupData = BackupData.get(server);
        if (!CommonConfig.isEnabled() || CommonConfig.backupsDisabledByJvmArg() || backupData.isPaused()) {
            return false;
        }
        if (CherishedWorldsCompat.isLoaded() && !CherishedWorldsCompat.isFavorite(server.storageSource.getLevelId())) {
            return false;
        }
        boolean bl = arePlayersOnline = !server.getPlayerList().getPlayers().isEmpty();
        if (CommonConfig.useTickCounter()) {
            long timer;
            long lastSaved;
            long gameTime = server.overworld().getGameTime();
            return gameTime - (lastSaved = backupData.getLastSaved()) >= (timer = CommonConfig.getTimer(arePlayersOnline) * 20L * 60L);
        }
        return System.currentTimeMillis() - CommonConfig.getTimer(arePlayersOnline) > backupData.getLastSaved();
    }

    public static void createBackup(MinecraftServer server, boolean quiet) {
        BackupThread thread = new BackupThread(server, quiet, null);
        thread.start();
    }

    public void deleteFiles() {
        if (this.manager.getChains().isEmpty()) {
            return;
        }
        int maxChains = CommonConfig.getBackupsToKeep();
        while (this.manager.getChains().size() > maxChains) {
            BackupChain chain = this.manager.getFirstChain();
            LOGGER.info("Deleting backup chain directory \"{}\"", (Object)chain.getParentFolder());
            this.manager.removeChain(chain);
        }
    }

    public void saveStorageSize() {
        try {
            while (this.manager.getFileSize() > CommonConfig.getMaxDiskSize()) {
                List<BackupChain> chains = this.manager.getChains();
                if (chains.size() <= 1) {
                    LOGGER.error("Cannot delete old chains to save disk space. Only one chain directory left!");
                    return;
                }
                BackupChain victim = chains.getFirst();
                LOGGER.info("Deleting backup chain directory \"{}\" to save disk space", (Object)victim.getParentFolder());
                this.manager.removeChain(victim);
            }
        }
        catch (NullPointerException e) {
            LOGGER.error("Cannot delete old files to save disk space", (Throwable)e);
        }
    }

    @Override
    public void run() {
        try {
            Path backupFilePath = this.getChainBackupFilePath();
            try {
                this.deleteFiles();
                Files.createDirectories(this.backupPath, new FileAttribute[0]);
                long start = System.currentTimeMillis();
                this.broadcast("simplebackups.backup_started", Style.EMPTY.withColor(ChatFormatting.GOLD), new Object[0]);
                BackupResult backupResult = this.makeWorldBackup(backupFilePath);
                long end = System.currentTimeMillis();
                String time = Timer.getTimer(end - start);
                this.saveStorageSize();
                boolean hasErrors = !this.errors.isEmpty();
                this.broadcast("simplebackups.backup_finished", Style.EMPTY.withColor(hasErrors ? ChatFormatting.YELLOW : ChatFormatting.GOLD), time, StorageSize.getFormattedSize(backupResult.fileSize), StorageSize.getFormattedSize(this.getOutputFolderSize()));
                if (hasErrors) {
                    MutableComponent erroredFiles = Component.literal((String)this.errors.stream().map(file -> "- " + file.toString()).collect(Collectors.joining("\n")));
                    this.broadcast("simplebackups.backup_errors", Style.EMPTY.withColor(ChatFormatting.RED).withHoverEvent((HoverEvent)new HoverEvent.ShowText((Component)erroredFiles)), this.errors.size());
                    LOGGER.error("Skipped {} files during backup because of errors:", (Object)this.errors.size());
                    for (Path failedFile : this.errors) {
                        LOGGER.error(" - {}", (Object)failedFile);
                    }
                }
            }
            catch (NotEnoughDiskSpaceException e) {
                this.broadcast("simplebackups.not_enough_space", Style.EMPTY.withColor(ChatFormatting.RED), new Object[0]);
                Files.deleteIfExists(backupFilePath);
            }
            catch (IOException e) {
                if (CommonConfig.deleteUnfinishedBackup()) {
                    this.broadcast("simplebackups.backup_failed_delete", Style.EMPTY.withColor(ChatFormatting.RED), new Object[0]);
                    Files.deleteIfExists(backupFilePath);
                } else {
                    this.broadcast("simplebackups.backup_failed_continue", Style.EMPTY.withColor(ChatFormatting.RED), new Object[0]);
                }
                SimpleBackups.LOGGER.error("Error backing up", (Throwable)e);
            }
        }
        catch (IOException e) {
            SimpleBackups.LOGGER.error("Error backing up", (Throwable)e);
        }
    }

    private Path getChainBackupFilePath() throws IOException {
        Path worldBackupDir = this.backupPath;
        Files.createDirectories(Files.exists(worldBackupDir, new LinkOption[0]) ? worldBackupDir.toRealPath(new LinkOption[0]) : worldBackupDir, new FileAttribute[0]);
        String baseName = LocalDateTime.now().format(FORMATTER);
        BackupChain latestChain = this.manager.getLatestChain();
        if (this.fullBackup || latestChain == null) {
            this.forceFullBackup = true;
            BackupChain chain = this.manager.createChain(baseName);
            return chain.getFullBackup();
        }
        return latestChain.createChild();
    }

    private long getOutputFolderSize() {
        long l;
        block9: {
            if (!Files.exists(this.backupPath, new LinkOption[0])) {
                return 0L;
            }
            Stream<Path> stream = Files.walk(this.backupPath, new FileVisitOption[0]);
            try {
                l = stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).mapToLong(path -> {
                    try {
                        return Files.size(path);
                    }
                    catch (IOException e) {
                        LOGGER.warn("Failed to get size of {}", path, (Object)e);
                        return 0L;
                    }
                }).sum();
                if (stream == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    LOGGER.warn("Failed to get size of backup folder", (Throwable)e);
                    return 0L;
                }
            }
            stream.close();
        }
        return l;
    }

    private void broadcast(String message, Style style, Object ... parameters) {
        SimpleBackups.LOGGER.info(String.format(FMLTranslations.getPattern((String)message, () -> message), parameters));
        if (CommonConfig.sendMessages() && !this.quiet) {
            this.server.execute(() -> this.server.getPlayerList().getPlayers().forEach(player -> {
                if (ServerConfig.messagesForEveryone() || player.permissions().hasPermission(Permissions.COMMANDS_GAMEMASTER)) {
                    player.sendSystemMessage((Component)BackupThread.component(player, message, parameters).withStyle(style));
                }
            }));
            if (Mc2DiscordCompat.isLoaded() && CommonConfig.mc2discord()) {
                Mc2DiscordCompat.announce((Component)BackupThread.component(null, message, parameters));
            }
        }
    }

    public static MutableComponent component(@Nullable ServerPlayer player, String key, Object ... parameters) {
        if (player != null && NetworkRegistry.hasChannel((Connection)player.connection.connection, null, (Identifier)Pause.ID)) {
            return Component.translatable((String)key, (Object[])parameters);
        }
        return Component.literal((String)String.format(FMLTranslations.getPattern((String)key, () -> key), parameters));
    }

    private BackupResult makeWorldBackup(Path outputFile) throws IOException {
        this.storageSource.checkLock();
        if (CommonConfig.saveAll()) {
            this.server.executeBlocking(() -> this.server.saveEverything(true, false, true));
        }
        try (final ZipOutputStream zipStream = new ZipOutputStream(new BufferedOutputStream(Files.newOutputStream(outputFile, new OpenOption[0])));){
            zipStream.setLevel(CommonConfig.getCompressionLevel());
            final Path levelName = Paths.get(this.storageSource.getLevelId(), new String[0]);
            final Path levelPath = this.storageSource.getWorldDir().resolve(this.storageSource.getLevelId()).toRealPath(new LinkOption[0]);
            final List<Path> ignoredPaths = CommonConfig.getIgnoredPaths();
            final List<Path> ignoredFiles = CommonConfig.getIgnoredFiles();
            final String ignoredFilesRegex = CommonConfig.getIgnoredFilesRegex();
            final boolean ignoreSomething = !ignoredPaths.isEmpty() || !ignoredFiles.isEmpty() || !ignoredFilesRegex.isEmpty();
            final FileStore fileStore = Files.getFileStore(outputFile);
            Files.walkFileTree(levelPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                @Nonnull
                public FileVisitResult visitFile(@Nonnull Path file, @Nonnull BasicFileAttributes attrs) throws IOException {
                    if (file.endsWith("session.lock")) {
                        return FileVisitResult.CONTINUE;
                    }
                    if (file.endsWith("biomancy.spatial.db")) {
                        SimpleBackups.LOGGER.info("Skipping \"{}\" - see https://github.com/Elenterius/Biomancy/issues/175", (Object)levelPath.relativize(file));
                        return FileVisitResult.CONTINUE;
                    }
                    if (ignoreSomething && this.shouldSkipFile(levelPath.relativize(file))) {
                        SimpleBackups.LOGGER.debug("Skipping file: {}", (Object)file);
                        return FileVisitResult.CONTINUE;
                    }
                    long lastModified = file.toFile().lastModified();
                    if (BackupThread.this.createFullBackup() || lastModified - BackupThread.this.lastSaved > 0L) {
                        if (fileStore.getUsableSpace() - attrs.size() - 0x8000000L < 0L) {
                            throw new NotEnoughDiskSpaceException("Not enough space on disk to create backup");
                        }
                        String completePath = levelName.resolve(levelPath.relativize(file)).toString().replace('\\', '/');
                        ZipEntry zipentry = new ZipEntry(completePath);
                        try (InputStream inputStream = Files.newInputStream(file, new OpenOption[0]);){
                            zipStream.putNextEntry(zipentry);
                            inputStream.transferTo(zipStream);
                            zipStream.closeEntry();
                        }
                        catch (IOException e) {
                            this.visitFileFailed(file, e);
                        }
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                @Nonnull
                public FileVisitResult visitFileFailed(@Nonnull Path file, @Nonnull IOException exc) throws IOException {
                    if (exc instanceof NoSuchFileException || exc instanceof FileNotFoundException) {
                        SimpleBackups.LOGGER.debug("Skipped vanished file: {}", (Object)file);
                        return FileVisitResult.CONTINUE;
                    }
                    if (CommonConfig.collectErrors()) {
                        SimpleBackups.LOGGER.error("Failed to backup file: {}", (Object)file, (Object)exc);
                        BackupThread.this.errors.add(levelPath.relativize(file));
                        return FileVisitResult.CONTINUE;
                    }
                    IOException detailedException = new IOException("Failed to backup file: " + String.valueOf(file), exc);
                    return super.visitFileFailed(file, detailedException);
                }

                private boolean shouldSkipFile(Path relativePath) {
                    return ignoredPaths.contains(relativePath.getParent()) || ignoredFiles.contains(relativePath) || !ignoredFilesRegex.isEmpty() && this.getNormalizedPath(relativePath).matches(ignoredFilesRegex);
                }

                private String getNormalizedPath(Path path) {
                    return path.toString().replace('\\', '/');
                }
            });
        }
        return new BackupResult(outputFile, Files.size(outputFile));
    }

    private boolean createFullBackup() {
        return this.fullBackup || this.forceFullBackup;
    }

    private record BackupResult(Path outputFile, long fileSize) {
    }

    private static class Timer {
        private static final SimpleDateFormat SECONDS = new SimpleDateFormat("s.SSS");
        private static final SimpleDateFormat MINUTES = new SimpleDateFormat("mm:ss");
        private static final SimpleDateFormat HOURS = new SimpleDateFormat("HH:mm");

        private Timer() {
        }

        public static String getTimer(long milliseconds) {
            Date date = new Date(milliseconds);
            double seconds = (double)milliseconds / 1000.0;
            if (seconds < 60.0) {
                return SECONDS.format(date) + "s";
            }
            if (seconds < 3600.0) {
                return MINUTES.format(date) + "min";
            }
            return HOURS.format(date) + "h";
        }
    }
}

